<?php

namespace Bespin\DocumentType;

enum DocumentType: string
{
    case ACCOUNT_STATEMENT = 'accountStatement';
    case BILL = 'bill';
    case CONTENT_IDENTIFICATION = 'contentIdentification';
    case CORRESPONDENCE = 'correspondence';
    case CREDIT = 'credit';
    case DATEV_HEALTH_INSURANCE = 'healthInsurance'; // Krankenkasse Beitragsabrechnung
    case DATEV_HEALTH_INSURANCE_ALLOCATION = 'healthInsuranceAllocation'; // Umlagebeitraege
    case DATEV_PAYROLL_ACCOUNTING = 'datevPayrollAccounting'; // Datev Lohnbuchhaltung
    case DATEV_PROTOCOL_PAYROLL_DECLARATION = 'protocolPayrollDeclaration'; // Datenuebertragungsprotokoll Lohnsteuer Anmeldung
    case DATEV_PROTOCOL_STATEMENT_OF_CONTRIBUTION = 'protocolStatementOfContribution'; // Datenuebertragungsprotokoll Beitragsnachweis
    case DATEV_WAGE_JOURNAL = 'wageJournal'; // Lohnjournal
    case DATEV_WAGE_JOURNAL_ANNUAL = 'wageJournalAnnual'; // Lohnjournal Jahresuebersicht
    case INVOICE = 'invoice';
    case INVOICE_ATTACHMENT = 'invoiceAttachment';
    case MIXED = 'mixed';
    case ORDER = 'order';
    case OTHER = 'other';
    case PAYROLL = 'payroll';
    case PURCHASE_ORDER = 'purchaseOrder';
    case QUOTE = 'quote';
    case REPORT = 'report';
    case REPORT_SIGNED = 'reportSigned';
    case SETTLEMENT = 'settlement';
    case SICK_NOTE = 'sickNote';
    case TEMP_UPLOAD = 'tempUpload';
    case UNDEFINED = 'undefined';
    case UNIDENTIFIED = 'unidentified';
    case UNUSED_ORIGINAL = 'unusedOriginal'; // for example when we split a document into several other documents, we need to preserve the original
    case WAGE_TAX = 'wageTax';
    case DATEV_SOCIAL_SECURITY_ANNUAL = 'socialSecurityAnnual';
    case DATEV_DEUEV_REGISTRATION = 'deUevRegistration';

    public function name(): string
    {
        return $this->value;
    }

    public function isUndefined(): bool
    {
        return match ($this) {
            self::UNDEFINED, self::UNIDENTIFIED, self::CONTENT_IDENTIFICATION, self::MIXED, self::TEMP_UPLOAD => true,
            default                                                                                           => false,
        };
    }

    public function getTypeId(): int
    {
        return match ($this) {
            self::INVOICE                                  => 1,
            self::SICK_NOTE                                => 2,
            self::REPORT                                   => 3,
            self::REPORT_SIGNED                            => 4,
            self::BILL                                     => 5,
            self::ACCOUNT_STATEMENT                        => 6,
            self::PAYROLL                                  => 7,
            self::PURCHASE_ORDER                           => 8,
            self::SETTLEMENT                               => 9,
            self::CORRESPONDENCE                           => 10,
            self::WAGE_TAX                                 => 11,
            self::OTHER                                    => 12,
            self::QUOTE                                    => 13,
            self::ORDER                                    => 14,
            self::INVOICE_ATTACHMENT                       => 15,
            self::DATEV_HEALTH_INSURANCE                   => 16,
            self::DATEV_HEALTH_INSURANCE_ALLOCATION        => 17,
            self::DATEV_WAGE_JOURNAL                       => 18,
            self::DATEV_WAGE_JOURNAL_ANNUAL                => 19,
            self::DATEV_PROTOCOL_STATEMENT_OF_CONTRIBUTION => 20,
            self::DATEV_PROTOCOL_PAYROLL_DECLARATION       => 21,
            self::DATEV_PAYROLL_ACCOUNTING                 => 22,
            self::CREDIT                                   => 23,
            self::UNUSED_ORIGINAL                          => 24,
            self::CONTENT_IDENTIFICATION                   => 25,
            self::MIXED                                    => 26,
            self::TEMP_UPLOAD                              => 27,
            self::UNDEFINED                                => 28,
            self::UNIDENTIFIED                             => 29,
            self::DATEV_SOCIAL_SECURITY_ANNUAL             => 30,
            self::DATEV_DEUEV_REGISTRATION                 => 31
        };
    }

    public static function tryFromId(int $typeId): ?self
    {
        return match ($typeId) {
            1       => self::INVOICE,
            2       => self::SICK_NOTE,
            3       => self::REPORT,
            4       => self::REPORT_SIGNED,
            5       => self::BILL,
            6       => self::ACCOUNT_STATEMENT,
            7       => self::PAYROLL,
            8       => self::PURCHASE_ORDER,
            9       => self::SETTLEMENT,
            10      => self::CORRESPONDENCE,
            11      => self::WAGE_TAX,
            12      => self::OTHER,
            13      => self::QUOTE,
            14      => self::ORDER,
            15      => self::INVOICE_ATTACHMENT,
            16      => self::DATEV_HEALTH_INSURANCE,
            17      => self::DATEV_HEALTH_INSURANCE_ALLOCATION,
            18      => self::DATEV_WAGE_JOURNAL,
            19      => self::DATEV_WAGE_JOURNAL_ANNUAL,
            20      => self::DATEV_PROTOCOL_STATEMENT_OF_CONTRIBUTION,
            21      => self::DATEV_PROTOCOL_PAYROLL_DECLARATION,
            22      => self::DATEV_PAYROLL_ACCOUNTING,
            23      => self::CREDIT,
            24      => self::UNUSED_ORIGINAL,
            25      => self::CONTENT_IDENTIFICATION,
            26      => self::MIXED,
            27      => self::TEMP_UPLOAD,
            29      => self::UNIDENTIFIED,
            30      => self::DATEV_SOCIAL_SECURITY_ANNUAL,
            31      => self::DATEV_DEUEV_REGISTRATION,
            default => self::UNDEFINED
        };
    }

    public static function get(string $type): self
    {
        return self::from($type);
    }
}
