<?php /** @noinspection PhpUnused */

namespace Bespin\DocumentType;

use JsonSerializable;
use OpenApi\Attributes\Items;
use OpenApi\Attributes\Property;
use OpenApi\Attributes\Schema;

#[Schema]
class Document implements JsonSerializable
{
    #[Property]
    public int     $documentId;
    #[Property]
    public ?string $documentDate;
    #[Property]
    public string  $documentType;
    #[Property]
    public ?string $documentSender;
    #[Property]
    public ?string $referenceNumber;
    #[Property]
    public int     $sourceId;
    #[Property]
    public int     $vendorEmailId;
    #[Property]
    public ?int    $employeeId;
    #[Property]
    public string  $createdOn;
    #[Property]
    public string  $md5;
    #[Property]
    public string  $url;
    #[Property]
    public bool    $unread;
    #[Property]
    public bool    $inspected;
    /** @var array<int, Tag> */
    #[Property(items: new Items(type: "string"))]
    public array         $tags = [];
    public ?string       $storageProvider;
    public ?string       $jsonTags;
    public ?DocumentType $type;
    public ?string       $jsonToDos;
    /** @var array<int, ToDo> */
    #[Property(items: new Items(type: "string"))]
    public array $toDos = [];
    public ?int  $pages = null;

    /**
     * @return array<string, mixed>
     */
    public function jsonSerialize(): array
    {
        /**
         * @var array{type: ?DocumentType, jsonTags: ?string, jsonToDos: ?string, toDos: array<int, ToDo>, storageProvider: ?string, tags: array<int, Tag>} $result
         */
        $result = get_object_vars($this);
        unset($result['storageProvider']);
        $result['documentType'] = $result['type']?->value ?? '';
        unset ($result['type']);
        if (!empty($result['jsonTags']) && $result['jsonTags'] !== '[null]' && empty($result['tags'])) {
            $tags = json_decode($result['jsonTags']);
            if (is_array($tags)) {
                $result['tags'] = $tags;
            }
        }
        unset($result['jsonTags']);
        if (!empty($result['jsonToDos']) && $result['jsonToDos'] !== '[null]' && empty($result['toDos'])) {
            $toDos = json_decode($result['jsonToDos']);
            if (is_array($toDos)) {
                $result['toDos'] = $toDos;
            }
        }
        unset($result['jsonToDos']);
        return $result;
    }

    /**
     * @phpstan-ignore-next-line
     */
    public static function create(array $data): ?self
    {
        if (empty($data)) {
            return null;
        }
        $self = new self();
        foreach ($data as $property => $value) {
            switch ($property) {
                case 'tags':
                    if (is_array($value)) {
                        foreach ($value as $tagProperties) {
                            if (!empty($tagProperties)) {
                                $tag = Tag::create($tagProperties);
                                if ($tag !== null) {
                                    $self->tags[$tag->tagId] = $tag;
                                }
                            }
                        }
                    }
                    break;
                case 'toDos':
                    if (is_array($value)) {
                        foreach ($value as $todoProperties) {
                            if (!empty($todoProperties)) {
                                $toDo = ToDo::create($todoProperties);
                                if ($toDo !== null) {
                                    $self->toDos[$toDo->todoId] = $toDo;
                                }
                            }
                        }
                    }
                    break;
                case 'documentType':
                    $self->documentType = $value;
                    $self->type         = DocumentType::tryFrom($value);
                    break;
                default:
                    $self->{$property} = $value;
                    break;
            }
        }
        return $self;
    }
}
