<?php /** @noinspection PhpUnused */

namespace Bespin\DocumentType;

use DateTime;
use OpenApi\Attributes\Schema;

#[Schema]
class Metadata
{
    private string   $raw = '';
    private float    $value;
    private float    $vatRegular;
    private float    $vatReduced;
    private DateTime $documentDate;
    private string   $invoiceNumber;
    private string   $vendor;

    public function getRaw(): string
    {
        return $this->raw;
    }

    public function getValue(): ?float
    {
        return $this->value ?? null;
    }

    public function getVatRegular(): ?float
    {
        return $this->vatRegular ?? null;
    }

    public function getVatReduced(): ?float
    {
        return $this->vatReduced ?? null;
    }

    public function getDocumentDate(): ?DateTime
    {
        return $this->documentDate ?? null;
    }

    public function getInvoiceNumber(): string
    {
        return $this->invoiceNumber ?? '';
    }

    public function getVendor(): string
    {
        return $this->vendor ?? '';
    }


    /**
     * @param object{analyzed: object, defined: object} $metadata
     * @return self|null
     */
    public static function create(object $metadata): ?self
    {
        $meta = $metadata->analyzed;
        //TODO multiple invoices in a single pdf, other means of multiple documents -> see count $meta->analyzeResult->documents
        $self = new self();
        $json = json_encode($meta);
        if ($json !== false) {
            $self->raw = $json;
        }

        if (isset($meta->status) && $meta->status === 'succeeded' && isset($meta->analyzeResult->documents) && is_array($meta->analyzeResult->documents) && count($meta->analyzeResult->documents) === 1) {
            //Hopefully an Azure ai result
            $analyzedDocument = $meta->analyzeResult->documents[0];
            if (isset($analyzedDocument->fields->InvoiceId)) {
                $self->invoiceNumber = $analyzedDocument->fields->InvoiceId->valueString;
            }
            if (isset($analyzedDocument->fields->InvoiceDate)) {
                $date = DateTime::createFromFormat('Y-m-d', $analyzedDocument->fields->InvoiceDate->valueDate);
                if ($date !== false) {
                    $self->documentDate = $date;
                }
            }
            if (isset($analyzedDocument->fields->InvoiceTotal)) {
                $self->value = $analyzedDocument->fields->InvoiceTotal->valueCurrency->amount;
            }
            if (isset($analyzedDocument->fields->TaxDetails)) {
                if ($analyzedDocument->fields->TaxDetails->type === 'array') {
                    foreach ($analyzedDocument->fields->TaxDetails->valueArray as $taxDetail) {
                        if (isset($taxDetail->valueObject)) {
                            if (isset($taxDetail->valueObject->Rate, $taxDetail->valueObject->Amount->valueCurrency->amount)) {
                                if ($taxDetail->valueObject->Rate->valueString === '19 %') {
                                    $self->vatRegular = $taxDetail->valueObject->Amount->valueCurrency->amount;
                                } elseif ($taxDetail->valueObject->Rate->valueString === '7 %') {
                                    $self->vatReduced = $taxDetail->valueObject->Amount->valueCurrency->amount;
                                }
                            }
                        } else {
                            if (str_contains($taxDetail->content, '19%')) {
                                $self->vatRegular = (float)$taxDetail->valueObject->Amount->valueCurrency->amount;
                            } elseif (str_contains($taxDetail->content, '7%')) {
                                $self->vatReduced = (float)$taxDetail->valueObject->Amount->valueCurrency->amount;
                            }
                        }
                    }
                }
            } elseif (isset($analyzedDocument->fields->TotalTax)) {
                $self->vatRegular = (float)$analyzedDocument->fields->TotalTax->valueCurrency->amount;
            }
            if (isset($analyzedDocument->fields->VendorName)) {
                $self->vendor = $analyzedDocument->fields->VendorName->valueString;
            }
        }
        if (!empty($metadata->defined->documentDate)) {
            $date = DateTime::createFromFormat('Y-m-d', $metadata->defined->documentDate);
            if ($date !== false) {
                $self->documentDate = $date;
            }
        }
        if (!empty($metadata->defined->referenceNumber)) {
            $self->invoiceNumber = $metadata->defined->referenceNumber;
        }
        if (!empty($metadata->defined->documentSender)) {
            $self->vendor = $metadata->defined->documentSender;
        }
        return $self;
    }
}