<?php

namespace Bespin\DocumentClient\Helper;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use CurlHandle;
use Exception;

class Rest
{
    public static bool $allowInsecure = false;

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function create(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            method     : 'POST',
            parameters : $parameters,
            headers    : $headers,
            associative: $associative
        );
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function read(string $endpoint, string $location = '', array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative
        );
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function update(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            method     : 'PUT',
            parameters : $parameters,
            headers    : $headers,
            associative: $associative,
            transform  : BodyTransform::URL_ENCODE
        );
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function delete(string $endpoint, string $location, array $parameters = [], array $headers = []): mixed
    {
        return self::curl(
            endpoint  : $endpoint,
            location  : $location,
            method    : 'DELETE',
            parameters: $parameters,
            headers   : $headers,
            transform : BodyTransform::URL_ENCODE);
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function curl(
        string        $endpoint,
        string        $location = '',
        string        $method = 'GET',
        array         $parameters = [],
        array         $headers = [],
        bool          $associative = false,
        BodyTransform $transform = BodyTransform::NO,
        bool          $returnHeaders = false): mixed
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        $target = $endpoint.$location;
        if ($method === 'GET') {
            if (!empty($parameters)) {
                $target .= '?'.http_build_query($parameters);
            }
        } else {
            $parameters = match ($transform) {
                BodyTransform::JSON_ENCODE => json_encode($parameters),
                BodyTransform::URL_ENCODE  => http_build_query($parameters),
                default                    => $parameters,
            };
            curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
        }
        curl_setopt($ch, CURLOPT_URL, $target);

        if (self::$allowInsecure === true) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        }
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }

        $responseHeaders = [];
        curl_setopt($ch, CURLOPT_HEADERFUNCTION, function (CurlHandle $curl, string $header) use (&$responseHeaders) {
            $headers = explode(':', $header, 2);
            if (count($headers) === 2) {
                $responseHeaders[strtolower(trim($headers[0]))] = trim($headers[1]);
            }
            return strlen($header);
        });

        $responseBody = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if ($returnHeaders === true) {
            return [$responseHeaders, self::parseResult($responseHeaders, $responseBody, $curlHttpCode, $errno, $endpoint.$location, $method, $associative)];
        }
        return self::parseResult($responseHeaders, $responseBody, $curlHttpCode, $errno, $endpoint.$location, $method, $associative);
    }


    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    private static function parseResult(array $responseHeaders, string $responseBody, int $httpCode, int $errorNumber, string $url, string $method, bool $associative = false): mixed
    {
        $responseCategory = $httpCode - $httpCode % 100;
        switch ($responseCategory) {
            case 200:
                if (array_key_exists('content-type', $responseHeaders)) {
                    [$mimeType, $additionalContentTypeData] = self::parseContentType($responseHeaders['content-type']);
                    $contentType = ContentType::tryFrom($mimeType);
                    switch ($contentType) {
                        case ContentType::JSON:
                            try {
                                return json_decode($responseBody, $associative);
                            } catch (Exception) {
                                throw new Exception('Failed to json decode api response: '.$result);
                            }
                        default:
                            return $responseBody;
                    }
                }
                return $responseBody;
            case 403:
                throw new PermissionDeniedException("CURL failed: \nurl: ".$url." \nmethod: '.$method.' \ncurl error number: ".$errorNumber." \n http code: ".$httpCode);
            case 404:
                throw new NotFoundException("CURL failed: \nurl: ".$url." \nmethod: '.$method.' \ncurl error number: ".$errorNumber." \n http code: ".$httpCode);
            default:
                throw new Exception("CURL failed: \nurl: ".$url." \nmethod: '.$method.' \ncurl error number: ".$errorNumber." \n http code: ".$httpCode);
        }
    }

    private static function parseContentType(string $contentType): array
    {
        $parts    = explode(';', $contentType);
        $mimeType = strtolower(trim(array_shift($parts)));  // Remove and retrieve the first element, the MIME type
        $params   = [];

        foreach ($parts as $part) {
            $pair = explode('=', $part, 2);
            if (count($pair) == 2) {
                $params[strtolower(trim($pair[0]))] = strtolower(trim($pair[1]));
            }
        }

        return [$mimeType, $params];
    }
}