<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use Exception;
use DateTimeInterface;
use Bespin\DocumentClient\Helper\Rest;
use Bespin\DocumentType\Document;
use Bespin\DocumentType\DocumentType;

class Documents
{
    private string $url;
    private string $bearer;

    public function __construct(string $url, string $bearer)
    {
        $this->url    = $url;
        $this->bearer = $bearer;
    }

    /**
     * @return array<int, Document>
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function getDocuments(
        int                $userId,
        array              $groupIds = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        array              $tags = [],
        ?int               $rangeFrom = null,
        ?int               $rangeTo = null,
        DocumentType       ...$types): array
    {
        $parameters = ['userId' => $userId];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        foreach ($types as $type) {
            $parameters['types'][] = $type->name();
        }
        if ($from !== null) {
            $parameters['from'] = $from->format('Y-m-d');
        }
        if ($to !== null) {
            $parameters['to'] = $to->format('Y-m-d');
        }
        if ($createdAfter !== null) {
            $parameters['createdAfter'] = $createdAfter->format('Y-m-d H:i:s');
        }
        if ($inbox === true) {
            $parameters['inbox'] = true;
        }
        if (!empty($tags)) {
            $parameters['tags'] = $tags;//json_encode($tags);
        }
        $headers = ['AUTHORIZATION: '.$this->bearer];
        if ($rangeFrom !== null) {
            $headers[] = 'RANGE: documents='.$rangeFrom.($rangeTo !== null ? '-'.$rangeTo : '');
        }
        $documents = Rest::read(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, associative: true);
        if (empty($documents)) {
            return $documents;
        }
        $result = [];
        foreach ($documents as $document) {
            $result[$document['documentId']] = Document::create($document);
        }
        return $result;
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function getNumberOfInboxDocuments(int $userId, array $groupIds = []): int
    {
        $parameters = [
            'userId' => $userId,
            'inbox'  => true];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        $headers = [
            'AUTHORIZATION: '.$this->bearer,
            'RANGE: documents=0'];
        [$response] = Rest::curl(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, returnHeaders: true);
        if (array_key_exists('content-range', $response)) {
            $range = explode('/', $response['content-range']);
            if (count($range) === 2) {
                return $range[1];
            }
        }
        return 0;
    }
}
