<?php

namespace Bespin\DocumentClient\Helper;

use Exception;

class PdfUtil
{
    /**
     * @return string
     * @throws Exception
     */
    private static function pdfUtil(): string
    {
        $binaryPath = realpath(__DIR__.'/../../bin/pdfutil');
        if ($binaryPath !== false && file_exists($binaryPath) && is_executable($binaryPath)) {
            return $binaryPath;
        }
        throw new Exception('PDF util not found');
    }

    public static function getNumberOfPages(string $filename): int
    {
        $arguments = [
            self::pdfUtil(),
            'number-of-pages',
            '--in', $filename,
        ];
        $result    = Shell::exec(
            $arguments
        );
        if ($result->getExitCode() === 0 && is_numeric($result->getStdOut())) {
            return (int)$result->getStdOut();
        }
        throw new Exception($result->getStdErr(), $result->getExitCode());
    }

    /**
     * @param string $inputFileName
     * @param string $outputDirectory
     * @param array<string|int> $documentPages each array element creates a new document. the value can be comma separated pages and/or dashed ranges like this: 1-5,7,9,11-15
     * @param string $outputPrefix
     * @return array<string> returns an array of the files which were created
     * @throws Exception
     */
    public static function splitPdf(string $inputFileName, string $outputDirectory, array $documentPages, string $outputPrefix = 'document_'): array
    {
        // remove any whitespace in the page ranges as our pdf utility doesn't support this
        $documentPages = array_map(function ($value) {
            return preg_replace('/\s+/', '', (string)$value);
        }, $documentPages);
        $arguments     = [
            self::pdfUtil(),
            'split',
            '--in', $inputFileName,
            '--dir', $outputDirectory,
            '--pages', implode(';', $documentPages),
            '--prefix', $outputPrefix,
            '--format', 'json'
        ];
        $result        = Shell::exec(
            $arguments
        );
        if ($result->getExitCode() === 0) {
            $json = json_decode($result->getStdOut());
            if (is_array($json)) {
                return $json;
            }
        }
        throw new Exception($result->getStdErr(), $result->getExitCode());
    }

    /**
     * @throws Exception
     */
    public static function mergePdfs(string $outputFilename, string ...$inputFilenames): bool
    {
        $arguments = [
            self::pdfUtil(),
            'merge',
            '--out', $outputFilename,
            '--in'
        ];
        foreach ($inputFilenames as $inputFilename) {
            $arguments[] = $inputFilename;
        }
        $result = Shell::exec(
            $arguments
        );
        return $result->getExitCode() === 0;
    }

    /**
     * @throws Exception
     */
    public static function getMetadata(string $filename): ?PdfMetadata
    {
        $arguments = [
            self::pdfUtil(),
            'metadata',
            'get',
            '--in', $filename,
            '--format', 'json'
        ];
        $result    = Shell::exec(
            $arguments
        );
        if ($result->getExitCode() === 0) {
            return PdfMetadata::create($result->getStdOut());
        } else {
            throw new Exception($result->getStdErr(), $result->getExitCode());
        }
    }

    /**
     * @throws Exception
     */
    public static function setMetadata(string $filename, ?PdfMetadata $metadata, ?string $outputFilename = null): bool
    {
        $data = $metadata?->getMetadata();
        if ($data !== null) {
            $arguments = [
                self::pdfUtil(),
                'metadata',
                'set',
                '--in', $filename,
                '--data', $data
            ];
            if ($outputFilename !== null) {
                $arguments[] = '--out';
                $arguments[] = $outputFilename;
            }
            $result = Shell::exec(
                $arguments
            );
            return $result->getExitCode() === 0;
        }
        return false;
    }
}