<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use Exception;
use DateTimeInterface;
use Bespin\DocumentClient\Helper\Rest;
use Bespin\DocumentType\Document;
use Bespin\DocumentType\DocumentType;

class Documents
{
    private string $url;
    private string $bearer;

    public function __construct(string $url, string $bearer)
    {
        $this->url    = $url;
        $this->bearer = $bearer;
    }

    /**
     * @param array<string> $groups
     * @param array<int, string> $tags
     * @return array<int, Document>
     * @throws Exception
     */
    public function getDocuments(
        int                $userId,
        array              $groups = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?int               $employeeId = null,
        array              $tags = [],
        ?int               $rangeFrom = null,
        ?int               $rangeTo = null,
        ?bool              $assigned = null,
        ?DocumentTypes     $types = null): array
    {
        $parameters = $this->getParameters($userId, $groups, $from, $to, $createdAfter, $inbox, $read, $todo, $assigned, $employeeId, $tags, ...($types?->getDocumentTypes() ?? []));
        $headers    = ['AUTHORIZATION: '.$this->bearer];
        if ($rangeFrom !== null) {
            $headers[] = 'RANGE: documents='.$rangeFrom.($rangeTo !== null ? '-'.$rangeTo : '');
        }
        try {
            $documents = Rest::read(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, associative: true);
        } catch (PermissionDeniedException|NotFoundException) {
            return [];
        }
        if (empty($documents)) {
            return [];
        }
        $result = [];
        if (is_array($documents)) {
            foreach ($documents as $apiDocument) {
                $document = Document::create($apiDocument);
                if ($document !== null) {
                    $result[$document->documentId] = $document;
                }
            }
        }
        return $result;
    }

    /**
     * @param array<string> $groups
     * @param array<int, string> $tags
     * @throws Exception
     */
    public function getNumberOfDocuments(
        int                $userId,
        array              $groups = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?bool              $assigned = null,
        ?int               $employeeId = null,
        array              $tags = [],
        ?DocumentTypes     $types = null): int
    {
        $parameters = $this->getParameters($userId, $groups, $from, $to, $createdAfter, $inbox, $read, $todo, $assigned, $employeeId, $tags, ...($types?->getDocumentTypes() ?? []));
        return $this->queryDocumentRange($parameters);
    }

    /**
     * @param array<string> $groups
     * @throws Exception
     */
    public function getNumberOfInboxDocuments(int $userId, array $groups = []): int
    {
        $parameters = [
            'userId' => $userId,
            'inbox'  => true,
            'groups' => $groups];
        return $this->queryDocumentRange($parameters);
    }

    /**
     * @param array<string, array<int, int|string>|bool|int|string> $parameters
     * @throws Exception
     */
    private function queryDocumentRange(array $parameters): int
    {
        $headers = [
            'AUTHORIZATION: '.$this->bearer,
            'RANGE: documents=0'
        ];
        try {
            $apiResult = Rest::curl(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, returnHeaders: true);
        } catch (PermissionDeniedException|NotFoundException) {
            return 0;
        }
        if (!is_array($apiResult)) {
            return 0;
        }
        [$response] = $apiResult;
        if (array_key_exists('content-range', $response)) {
            $range = explode('/', $response['content-range']);
            if (count($range) === 2) {
                return (int)$range[1];
            }
        }
        return 0;
    }

    /**
     * @param array<string> $groups
     * @param array<int, string> $tags
     * @return array<string, array<int, int|string>|bool|int|string>
     */
    private function getParameters(
        int                $userId,
        array              $groups = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?bool              $assigned = null,
        ?int               $employeeId = null,
        array              $tags = [],
        DocumentType       ...$types): array
    {
        $parameters = ['userId' => $userId, 'groups' => $groups];
        if ($from !== null) {
            $parameters['from'] = $from->format('Y-m-d');
        }
        if ($to !== null) {
            $parameters['to'] = $to->format('Y-m-d');
        }
        if ($createdAfter !== null) {
            $parameters['createdAfter'] = $createdAfter->format('Y-m-d H:i:s');
        }
        if ($inbox === true) {
            $parameters['inbox'] = true;
        }
        if ($read !== null) {
            $parameters['read'] = $read;
        }
        if ($todo !== null) {
            $parameters['todo'] = $todo;
        }
        if ($assigned !== null) {
            $parameters['assigned'] = $assigned;
        }
        if ($employeeId !== null) {
            $parameters['employeeId'] = $employeeId;
        }
        if (!empty($tags)) {
            $parameters['tags'] = $tags;//json_encode($tags);
        }
        foreach ($types as $type) {
            $parameters['types'][] = $type->name();
        }
        return $parameters;
    }
}
