<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use Exception;
use DateTimeInterface;
use Bespin\DocumentClient\Helper\Rest;
use Bespin\DocumentType\Document;
use Bespin\DocumentType\DocumentType;

class Documents
{
    private string $url;
    private string $bearer;

    public function __construct(string $url, string $bearer)
    {
        $this->url    = $url;
        $this->bearer = $bearer;
    }

    /**
     * @param array<int> $groupIds
     * @param array<int, string> $tags
     * @return array<int, Document>
     * @throws Exception
     */
    public function getDocuments(
        int                $userId,
        array              $groupIds = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?int               $employeeId = null,
        array              $tags = [],
        ?int               $rangeFrom = null,
        ?int               $rangeTo = null,
        DocumentType       ...$types): array
    {
        $parameters = $this->getParameters($userId, $groupIds, $from, $to, $createdAfter, $inbox, $read, $todo, $employeeId, $tags, ...$types);
        $headers    = ['AUTHORIZATION: '.$this->bearer];
        if ($rangeFrom !== null) {
            $headers[] = 'RANGE: documents='.$rangeFrom.($rangeTo !== null ? '-'.$rangeTo : '');
        }
        try {
            $documents = Rest::read(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, associative: true);
        } catch (PermissionDeniedException|NotFoundException) {
            return [];
        }
        if (empty($documents)) {
            return [];
        }
        $result = [];
        if (is_array($documents)) {
            foreach ($documents as $apiDocument) {
                $document = Document::create($apiDocument);
                if ($document !== null) {
                    $result[$document->documentId] = $document;
                }
            }
        }
        return $result;
    }

    /**
     * @param array<int> $groupIds
     * @param array<int, string> $tags
     * @return array<string, array<int, int|string>|bool|int|string>
     */
    private function getParameters(
        int                $userId,
        array              $groupIds = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?int               $employeeId = null,
        array              $tags = [],
        DocumentType       ...$types): array
    {
        $parameters = ['userId' => $userId];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        if ($from !== null) {
            $parameters['from'] = $from->format('Y-m-d');
        }
        if ($to !== null) {
            $parameters['to'] = $to->format('Y-m-d');
        }
        if ($createdAfter !== null) {
            $parameters['createdAfter'] = $createdAfter->format('Y-m-d H:i:s');
        }
        if ($inbox === true) {
            $parameters['inbox'] = true;
        }
        if ($read !== null) {
            $parameters['read'] = $read;
        }
        if ($todo !== null) {
            $parameters['todo'] = $todo;
        }
        if ($employeeId !== null) {
            $parameters['employeeId'] = $employeeId;
        }
        if (!empty($tags)) {
            $parameters['tags'] = $tags;//json_encode($tags);
        }
        foreach ($types as $type) {
            $parameters['types'][] = $type->name();
        }
        return $parameters;
    }

    /**
     * @param array<int> $groupIds
     * @param array<int, string> $tags
     * @throws Exception
     */
    public function getNumberOfDocuments(
        int                $userId,
        array              $groupIds = [],
        ?DateTimeInterface $from = null,
        ?DateTimeInterface $to = null,
        ?DateTimeInterface $createdAfter = null,
        ?bool              $inbox = false,
        ?bool              $read = null,
        ?bool              $todo = null,
        ?int               $employeeId = null,
        array              $tags = [],
        DocumentType       ...$types): int
    {
        $parameters = $this->getParameters($userId, $groupIds, $from, $to, $createdAfter, $inbox, $read, $todo, $employeeId, $tags, ...$types);
        $headers    = [
            'AUTHORIZATION: '.$this->bearer,
            'RANGE: documents=0'
        ];
        try {
            $apiResult = Rest::read(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, associative: true);
        } catch (PermissionDeniedException|NotFoundException) {
            return 0;
        }
        if (!is_array($apiResult)) {
            return 0;
        }
        [$response] = $apiResult;
        if (array_key_exists('content-range', $response)) {
            $range = explode('/', $response['content-range']);
            if (count($range) === 2) {
                return (int)$range[1];
            }
        }
        return 0;
    }

    /**
     * @param array<int> $groupIds
     * @throws Exception
     */
    public function getNumberOfInboxDocuments(int $userId, array $groupIds = []): int
    {
        $parameters = [
            'userId' => $userId,
            'inbox'  => true];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        $headers = [
            'AUTHORIZATION: '.$this->bearer,
            'RANGE: documents=0'];
        try {
            $apiResult = Rest::curl(endpoint: $this->url, location: 'documents', parameters: $parameters, headers: $headers, returnHeaders: true);

        } catch (PermissionDeniedException|NotFoundException) {
            return 0;
        }
        if (!is_array($apiResult)) {
            return 0;
        }
        [$response] = $apiResult;
        if (array_key_exists('content-range', $response)) {
            $range = explode('/', $response['content-range']);
            if (count($range) === 2) {
                return (int)$range[1];
            }
        }
        return 0;
    }
}
