<?php

namespace Bespin\DocumentClient\File;

use Bespin\DocumentClient\Helper\Shell;
use CURLFile;
use CURLStringFile;
use DateTime;
use Exception;
use Smalot\PdfParser\Document;
use Smalot\PdfParser\Parser;

class PdfFile implements FileInterface
{
    private string  $fileName;
    private ?string $localFileName;
    /** @var array<string, string> */
    private array     $pathInfo;
    private ?int      $parentPage;
    private string    $qpdf         = '/usr/bin/qpdf';
    private ?Document $parsedDocument;
    private ?DateTime $documentDate = null;

    public function __construct(string $fileName, ?int $parentPage = null)
    {
        $this->fileName = $fileName;
        if (file_exists($fileName)) {
            $this->pathInfo = pathinfo($fileName);
        }
        $this->parentPage = $parentPage;
    }

    private function getLocalFileName(): string
    {
        if (!isset($this->localFileName)) {
            if (str_starts_with($this->fileName, 'https://') || str_starts_with($this->fileName, 'http:/')) {
                $baseName = basename($this->fileName);
                $tmpDir   = sys_get_temp_dir();
                if (!file_exists($tmpDir.'/'.$baseName)) {
                    file_put_contents($tmpDir.'/'.$baseName, file_get_contents($this->fileName));
                }
                $this->localFileName = $tmpDir.'/'.$baseName;
            } else {
                $this->localFileName = $this->fileName;
            }
        }
        return $this->localFileName;
    }

    private function getParsedDocument(): ?Document
    {
        if (!isset($this->parsedDocument)) {
            $parser = new Parser();
            try {
                $this->parsedDocument = $parser->parseFile($this->fileName);
            } catch (Exception) {
                $this->parsedDocument = null;
            }
        }
        return $this->parsedDocument;
    }

    public function getDocumentDate(): ?DateTime
    {
        if ($this->documentDate === null) {
            $details = $this->getParsedDocument()?->getDetails() ?? [];
            try {
                if (array_key_exists('CreationDate', $details)) {
                    $this->documentDate = new DateTime($details['CreationDate']);
                }
            } catch (Exception) {
                return $this->documentDate;
            }
        }
        return $this->documentDate;
    }

    public function setDocumentDate(?DateTime $documentDate): void
    {
        $this->documentDate = $documentDate;
    }

    public function getText(): string
    {
        return $this->getParsedDocument()?->getText() ?? '';
    }

    public function getDocumentTitle(): string
    {
        return array_key_exists('filename', $this->pathInfo) ? $this->pathInfo['filename'] : basename($this->fileName);
    }

    /**
     * @return int
     * @throws Exception
     */
    public function getNumberOfPages(): int
    {
        $numberOfPages = Shell::exec($this->qpdf.' --show-npages '.$this->getLocalFileName());
        if ($numberOfPages->getExitCode() !== 0) {
            throw new Exception($numberOfPages->getStdErr());
        }
        return (int)$numberOfPages->getStdOut();
    }

    /**
     * @return PdfFile[]
     * @throws Exception
     */
    public function splitDocumentGetPages(): array
    {
        $now           = new DateTime();
        $name          = md5($this->getLocalFileName().$now->format('Y-m-d_H:i:s'));
        $numberOfPages = $this->getNumberOfPages();
        $tmpDir        = sys_get_temp_dir();
        Shell::exec($this->qpdf.' '.$this->getLocalFileName().' '.$tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf --split-pages');
        $result = [];
        for ($i = 1; $i <= $numberOfPages; $i++) {
            if ($i < 10) {
                rename($tmpDir.DIRECTORY_SEPARATOR.$name.'-0'.$i.'.pdf', $tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf');
            }
            $result[$i] = new PdfFile($tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf', $i);
        }
        return $result;
    }

    public function getFileName(): string
    {
        return $this->fileName;
    }

    public function getParentPageNumber(): ?int
    {
        return $this->parentPage;
    }

    public function getCurlFile(): CURLFile|CURLStringFile
    {
        if (!file_exists($this->getFileName())) {
            throw new Exception('File '.$this->getFileName().' does not exist');
        }
        $mimeType = mime_content_type($this->getFileName());
        return new CURLFile($this->getFileName(), $mimeType === false ? null : $mimeType);
    }

    /**
     * @param array<int, int|PdfFile> $pages
     * @return PdfFile
     * @throws Exception
     */
    public function getPageRange(array $pages): PdfFile
    {
        $pageNumbers = [];
        foreach ($pages as $page) {
            if (is_int($page)) {
                $pageNumbers[] = $page;
            } elseif ($page instanceof PdfFile) {
                $pageNumbers[] = $page->parentPage;
            }
        }
        $now        = new DateTime();
        $name       = md5($this->getLocalFileName().$now->format('Y-m-d_H:i:s').implode('-', $pageNumbers));
        $tmpDir     = sys_get_temp_dir();
        $outputFile = $tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf';
        Shell::exec($this->qpdf.' '.$this->getLocalFileName().' --pages . '.implode(',', $pageNumbers).' -- '.$outputFile);
        return new PdfFile($outputFile);
    }

    public function __debugInfo(): array
    {
        $debug_info['parsedDocument'] = !isset($this->parsedDocument) ? '' : '**RECURSION**';
        return $debug_info;
    }

    /**
     * @return array<string, mixed>
     */
    public function jsonSerialize(): array
    {
        return $this->__debugInfo();
    }
}
