<?php

namespace Bespin\DocumentClient\File;

use DateTime;
use Exception;
use Smalot\PdfParser\Document;
use Smalot\PdfParser\Parser;
use stdClass;

class PdfFile implements FileInterface
{
    private string $fileName;
    private array $pathInfo;
    private ?int      $parentPage;
    private string    $qpdf = '/usr/bin/qpdf';
    private ?Document $parsedDocument;
    private ?DateTime $documentDate = null;

    public function __construct(string $fileName, ?int $parentPage = null)
    {
        $this->fileName = $fileName;
        if (file_exists($fileName)) {
            $this->pathInfo = pathinfo($fileName);
        }
        $this->parentPage = $parentPage;
    }

    private function getParsedDocument(): ?Document
    {
        if (!isset($this->parsedDocument)) {
            $parser               = new Parser();
            try {
                $this->parsedDocument = $parser->parseFile($this->fileName);
            } catch (Exception $e) {
                //Debug::error('Could not parse PDF. '.$e->getMessage());
                $this->parsedDocument = null;
            }
        }
        return $this->parsedDocument;
    }

    public function getDocumentDate(): ?DateTime
    {
        if ($this->documentDate === null) {
            $details = $this->getParsedDocument()?->getDetails() ?? [];
            try {
                if (array_key_exists('CreationDate', $details)) {
                    $this->documentDate = new DateTime($details['CreationDate']);
                }
            } catch (Exception) {
                return $this->documentDate;
            }
        }
        return $this->documentDate;
    }

    public function setDocumentDate(DateTime $documentDate)
    {
        $this->documentDate = $documentDate;
    }

    public function getText(): string
    {
        return $this->getParsedDocument()?->getText() ?? '';
    }

    public function getDocumentTitle(): string
    {
        return array_key_exists('filename', $this->pathInfo) ? $this->pathInfo['filename'] : basename($this->fileName);
    }

    public function getNumberOfPages(): int
    {
        $numberOfPages = $this->exec($this->qpdf.' --show-npages '.$this->fileName);
        if ($numberOfPages->exit !== 0) {
            print_r($numberOfPages);
        }
        return $numberOfPages->stdout;
    }

    /**
     * @return PdfFile[]
     */
    public function splitDocumentGetPages(): array
    {
        $now           = new DateTime();
        $name          = md5($this->fileName.$now->format('Y-m-d_H:i:s'));
        $numberOfPages = $this->getNumberOfPages();
        $tmpDir        = sys_get_temp_dir();
        $this->exec($this->qpdf.' '.$this->fileName.' '.$tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf --split-pages');
        $result = [];
        for ($i = 1; $i <= $numberOfPages; $i++) {
            if ($i < 10) {
                rename($tmpDir.DIRECTORY_SEPARATOR.$name.'-0'.$i.'.pdf', $tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf');
            }
            $result[$i] = new PdfFile($tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf', $i);
        }
        return $result;
    }

    public function getFileName(): string
    {
        return $this->fileName;
    }

    public function getParentPageNumber(): ?int
    {
        return $this->parentPage;
    }

    public function getPageRange(array $pages): PdfFile
    {
        $pageNumbers = [];
        foreach ($pages as $page) {
            if (is_int($page)) {
                $pageNumbers[] = $page;
            } elseif ($page instanceof PdfFile) {
                $pageNumbers[] = $page->parentPage;
            }
        }
        $now    = new DateTime();
        $name   = md5($this->fileName.$now->format('Y-m-d_H:i:s'));
        $tmpDir = sys_get_temp_dir();
        $this->exec($this->qpdf.' '.$this->fileName.' --pages . '.implode(' . ', $pageNumbers).' -- '.$tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf');
        return new PdfFile($tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf');
    }

    private function exec(string $command): object
    {
        $proc   = proc_open($command, [
            1 => ['pipe', 'w'],
            2 => ['pipe', 'w'],
        ], $pipes);
        $stdout = stream_get_contents($pipes[1]);
        fclose($pipes[1]);
        $stderr = stream_get_contents($pipes[2]);
        fclose($pipes[2]);
        $result          = new stdClass();
        $result->exit    = proc_close($proc);
        $result->stdout  = rtrim($stdout);
        $result->stderr  = rtrim($stderr);
        $result->command = $command;
        return $result;
    }
}
