<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use Bespin\DocumentClient\Helper\BodyTransform;
use Bespin\DocumentClient\Helper\Rest;
use Exception;

class Pdf
{
    /**
     * @param string $url the url of the pdf service
     */
    public function __construct(private readonly string $url)
    {
    }

    /**
     * @param string $url the url of the website of which a pdf should be generated
     * @param array<int, string> $getParameters
     * @return string
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function getPdf(string $url, array $getParameters = []): string
    {
        $parameters['url'] = $url;
        if (!empty($getParameters)) {
            $parameters['get_parameters'] = $getParameters;
        }

        $content       = json_encode($parameters);
        $contentLength = $content === false ? 0 : strlen($content);
        $headers       = [
            'Content-Length: '.$contentLength,
            'Content-Type: application/json'
        ];
        $apiResult     = Rest::curl(
            endpoint:   $this->url,
            method:     'POST',
            parameters: $parameters,
            headers:    $headers,
            transform:  BodyTransform::JSON_ENCODE
        );
        return is_string($apiResult) ? $apiResult : '';
    }

    /**
     * @param string $url the url of the website of which a pdf should be generated
     * @param string|null $zugferdXml optional ZUGFeRD XML to create a valid ZUGFeRD invoice
     * @param array<int, string> $getParameters
     * @return string
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @deprecated use getPdf instead
     */
    public function getZugferdPdf(string $url, ?string $zugferdXml = null, array $getParameters = []): string
    {
        trigger_error('Using getZugferdPdf is deprecated. Use getPdf instead', E_USER_DEPRECATED);
        return $this->getPdf($url, $getParameters);
    }
}
