<?php

namespace Bespin\DocumentClient\Parser;

use Bespin\DocumentClient\DocumentType\Document;
use Bespin\DocumentClient\File\FileInterface;
use Bespin\DocumentClient\DocumentType\Bill;
use Bespin\DocumentClient\Model\ParserModelInterface;
use Bespin\DocumentType\DocumentType;

class DocumentParser
{
    private DocumentType  $documentType;
    private FileInterface $pdfHandler;

    public function __construct(FileInterface $pdf, DocumentType $documentType = DocumentType::UNDEFINED)
    {
        $this->pdfHandler   = $pdf;
        $this->documentType = $documentType;
    }

    public function determineDocumentContent(): DocumentType
    {
        switch ($this->documentType) {
            case DocumentType::UNDEFINED:
            case DocumentType::MIXED:
                $textContent            = $this->pdfHandler->getText();
                $compressedLowerContent = strtolower(preg_replace('/\s*/m', '', $textContent));
                if (str_contains($compressedLowerContent, 'datev') || str_contains($compressedLowerContent, 'lohnsteuerbescheinigung')) {
                    return DocumentType::DATEV_PAYROLL_ACCOUNTING;
                }
                if (str_contains($compressedLowerContent, 'receipt') || str_contains($compressedLowerContent, 'invoice') || str_contains($compressedLowerContent, 'rechnung')) {
                    return DocumentType::BILL;
                }
                break;
            default:
                return $this->documentType;
        }
        return DocumentType::UNDEFINED;
    }

    public function getDocuments(ParserModelInterface $parserModel, string $sender, string $reference): array
    {
        $types  = $this->determineDocumentContent();
        $result = [];
        switch ($types) {
            case DocumentType::DATEV_PAYROLL_ACCOUNTING:
                $datevParser = new DatevParser($this->pdfHandler);
                return $datevParser->getDocuments($parserModel);
            case DocumentType::BILL:
                $result[] = new Bill($this->pdfHandler);
                break;
            default:
                $result[] = new Document($this->pdfHandler);
                break;
        }
        foreach ($result as $document) {
            $document->setSender($sender);
            $document->setReference($reference);
        }
        return $result;
    }
}