<?php

namespace Bespin\DocumentClient\File;

use DateTime;
use Exception;
use Smalot\PdfParser\Document;
use Smalot\PdfParser\Parser;
use stdClass;

class PdfFile implements FileInterface
{
    private string    $fileName;
    private ?string   $localFileName;
    private array     $pathInfo;
    private ?int      $parentPage;
    private string    $qpdf         = '/usr/bin/qpdf';
    private ?Document $parsedDocument;
    private ?DateTime $documentDate = null;

    public function __construct(string $fileName, ?int $parentPage = null)
    {
        $this->fileName = $fileName;
        if (file_exists($fileName)) {
            $this->pathInfo = pathinfo($fileName);
        }
        $this->parentPage = $parentPage;
    }

    private function getLocalFileName(): string
    {
        if ($this->localFileName === null) {
            if (str_starts_with($this->fileName, 'https://') || str_starts_with($this->fileName, 'http://')) {
                $baseName = basename($this->fileName);
                $tmpDir   = sys_get_temp_dir();
                if (!file_exists($tmpDir.'/'.$baseName)) {
                    file_put_contents($tmpDir.'/'.$baseName, file_get_contents($this->fileName));
                }
                $this->localFileName = $tmpDir.'/'.$baseName;
            } else {
                $this->localFileName = $this->fileName;
            }
        }
        return $this->localFileName;
    }

    private function getParsedDocument(): ?Document
    {
        if (!isset($this->parsedDocument)) {
            $parser = new Parser();
            try {
                $this->parsedDocument = $parser->parseFile($this->fileName);
            } catch (Exception) {
                $this->parsedDocument = null;
            }
        }
        return $this->parsedDocument;
    }

    public function getDocumentDate(): ?DateTime
    {
        if ($this->documentDate === null) {
            $details = $this->getParsedDocument()?->getDetails() ?? [];
            try {
                if (array_key_exists('CreationDate', $details)) {
                    $this->documentDate = new DateTime($details['CreationDate']);
                }
            } catch (Exception) {
                return $this->documentDate;
            }
        }
        return $this->documentDate;
    }

    public function setDocumentDate(DateTime $documentDate): void
    {
        $this->documentDate = $documentDate;
    }

    public function getText(): string
    {
        return $this->getParsedDocument()?->getText() ?? '';
    }

    public function getDocumentTitle(): string
    {
        return array_key_exists('filename', $this->pathInfo) ? $this->pathInfo['filename'] : basename($this->fileName);
    }

    /**
     * @return int
     * @throws Exception
     */
    public function getNumberOfPages(): int
    {
        $numberOfPages = $this->exec($this->qpdf.' --show-npages '.$this->getLocalFileName());
        if ($numberOfPages->exit !== 0) {
            throw new Exception($numberOfPages->stderr);
        }
        return $numberOfPages->stdout;
    }

    /**
     * @return PdfFile[]
     * @throws Exception
     */
    public function splitDocumentGetPages(): array
    {
        $now           = new DateTime();
        $name          = md5($this->getLocalFileName().$now->format('Y-m-d_H:i:s'));
        $numberOfPages = $this->getNumberOfPages();
        $tmpDir        = sys_get_temp_dir();
        $this->exec($this->qpdf.' '.$this->getLocalFileName().' '.$tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf --split-pages');
        $result = [];
        for ($i = 1; $i <= $numberOfPages; $i++) {
            if ($i < 10) {
                rename($tmpDir.DIRECTORY_SEPARATOR.$name.'-0'.$i.'.pdf', $tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf');
            }
            $result[$i] = new PdfFile($tmpDir.DIRECTORY_SEPARATOR.$name.'-'.$i.'.pdf', $i);
        }
        return $result;
    }

    public function getFileName(): string
    {
        return $this->fileName;
    }

    public function getParentPageNumber(): ?int
    {
        return $this->parentPage;
    }

    public function getPageRange(array $pages): PdfFile
    {
        $pageNumbers = [];
        foreach ($pages as $page) {
            if (is_int($page)) {
                $pageNumbers[] = $page;
            } elseif ($page instanceof PdfFile) {
                $pageNumbers[] = $page->parentPage;
            }
        }
        $now        = new DateTime();
        $name       = md5($this->getLocalFileName().$now->format('Y-m-d_H:i:s').implode('-', $pageNumbers));
        $tmpDir     = sys_get_temp_dir();
        $outputFile = $tmpDir.DIRECTORY_SEPARATOR.$name.'.pdf';
        $this->exec($this->qpdf.' '.$this->getLocalFileName().' --pages . '.implode(',', $pageNumbers).' -- '.$outputFile);
        return new PdfFile($outputFile);
    }

    private function exec(string $command): object
    {
        $proc   = proc_open($command, [
            1 => ['pipe', 'w'],
            2 => ['pipe', 'w'],
        ], $pipes);
        $stdout = stream_get_contents($pipes[1]);
        fclose($pipes[1]);
        $stderr = stream_get_contents($pipes[2]);
        fclose($pipes[2]);
        $result          = new stdClass();
        $result->exit    = proc_close($proc);
        $result->stdout  = rtrim($stdout);
        $result->stderr  = rtrim($stderr);
        $result->command = $command;
        return $result;
    }

    public function __debugInfo(): array
    {
        $debug_info['parsedDocument'] = !isset($this->parsedDocument) ? '' : '**RECURSION**';
        return $debug_info;
    }

    public function jsonSerialize(): array
    {
        return $this->__debugInfo();
    }
}
