<?php

namespace Bespin\DocumentClient\DocumentType;

use Bespin\DocumentClient\File\FileInterface;
use Bespin\DocumentType\DocumentType;
use CURLFile;
use CURLStringFile;
use DateTime;
use Exception;

class Document
{
    private string         $sender       = '';
    private string         $reference    = '';
    private ?FileInterface $document     = null;
    private string         $fileContents;
    private DocumentType   $documentType = DocumentType::UNDEFINED;
    private bool           $upload       = true;
    private bool           $assigned     = false;
    private ?DateTime      $documentDate = null;

    public function __construct(FileInterface|string $document, private readonly int $ownerId)
    {
        if ($document instanceof FileInterface) {
            $this->document = $document;
        } else {
            $this->fileContents = $document;
        }
    }

    public function getOwnerId(): int
    {
        return $this->ownerId;
    }

    public function setDocumentType(DocumentType $documentType): self
    {
        $this->documentType = $documentType;
        return $this;
    }

    public function getFile(): string
    {
        return $this->document?->getFileName() ?? '';
    }

    public function getType(): DocumentType
    {
        return $this->documentType;
    }

    public function getSender(): string
    {
        return $this->sender;
    }

    public function getReference(): string
    {
        return $this->reference;
    }

    public function getDocumentDate(): ?DateTime
    {
        return $this->documentDate;
    }

    public function setDocumentDate(?DateTime $date): self
    {
        if ($date !== null) {
            $this->documentDate = $date;
        }
        return $this;
    }

    /**
     * @return array<string>
     */
    public function getTags(): array
    {
        return [];
    }

    public function setReference(string $reference): void
    {
        $this->reference = $reference;
    }

    public function setSender(string $sender): void
    {
        $this->sender = $sender;
    }

    public function getUpload(): bool
    {
        return $this->upload;
    }

    public function getAssigned(): bool
    {
        return $this->assigned;
    }

    public function setAssigned(bool $assigned): void
    {
        $this->assigned = $assigned;
    }

    public function setUpload(bool $upload): void
    {
        $this->upload = $upload;
    }

    public function getCurlFile(): CURLFile|CURLStringFile
    {
        if (isset($this->fileContents)) {
            return new CURLStringFile($this->fileContents, '');
        }
        if (!file_exists($this->getFile())) {
            throw new Exception('File '.$this->getFile().' does not exist');
        }
        $mimeType = mime_content_type($this->getFile());
        return new CURLFile($this->getFile(), $mimeType === false ? null : $mimeType);
    }
}