<?php

namespace Bespin\DocumentClient\Helper;

use CurlHandle;
use Exception;

class Rest
{
    public static bool $allowInsecure = false;

    /** @throws Exception */
    public static function create(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curlWithPostFields(
            method     : 'POST',
            url        : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative
        );
    }

    /** @throws Exception */
    public static function read(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::get(
            url        : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative);
    }

    /** @throws Exception */
    public static function update(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curlWithPostFields(
            method     : 'PUT',
            url        : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative,
            transform  : BodyTransform::URL_ENCODE);
    }

    /** @throws Exception */
    public static function delete(string $endpoint, string $location, array $parameters = [], array $headers = []): mixed
    {
        return self::curlWithPostFields(
            method    : 'DELETE',
            url       : $endpoint,
            location  : $location,
            parameters: $parameters,
            headers   : $headers,
            transform : BodyTransform::URL_ENCODE);
    }

    /**
     * @throws Exception
     */
    public static function curlWithPostFields(string $method, string $url, string $location, array $parameters = [], array $headers = [], bool $associative = false, BodyTransform $transform = BodyTransform::NO): mixed
    {
        return self::curl(
            url        : $url,
            location   : $location,
            method     : $method,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative,
            transform  : $transform);
    }

    /** @throws Exception */
    public static function get(string $url, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            url        : $url,
            location   : $location,
            method     : 'GET',
            parameters : $parameters,
            headers    : $headers,
            associative: $associative);
    }

    /** @throws Exception */
    private static function curl(string $url, string $location, string $method, array $parameters = [], array $headers = [], bool $associative = false, BodyTransform $transform = BodyTransform::NO): mixed
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        $target = $url.$location;
        if ($method === 'GET') {
            if (!empty($parameters)) {
                $target .= '?'.http_build_query($parameters);
            }
        } else {
            $parameters = match ($transform) {
                BodyTransform::JSON_ENCODE => json_encode($parameters),
                BodyTransform::URL_ENCODE  => http_build_query($parameters),
                default                    => $parameters,
            };
            curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
        }
        curl_setopt($ch, CURLOPT_URL, $target);

        if (self::$allowInsecure === true) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        }
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }

        $responseHeaders = [];
        curl_setopt($ch, CURLOPT_HEADERFUNCTION, function (CurlHandle $curl, string $header) use (&$responseHeaders) {
            $headers = explode(':', $header, 2);
            if (count($headers) === 2) {
                $responseHeaders[strtolower(trim($headers[0]))] = trim($headers[1]);
            }
            return strlen($header);
        });

        $responseBody = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return self::parseResult($responseHeaders, $responseBody, $curlHttpCode, $errno, $url.$location, $method, $associative);
    }

    /**@throws Exception */
    private static function parseResult(array $responseHeaders, string $responseBody, int $httpCode, int $errorNumber, string $url, string $method, bool $associative = false): mixed
    {
        $responseCategory = $httpCode - $httpCode % 100;
        switch ($responseCategory) {
            case 200:
                if (array_key_exists('content-type', $responseHeaders)) {
                    $contentType = ContentType::tryFrom(strtolower($responseHeaders['content-type']));
                    switch ($contentType) {
                        case ContentType::JSON:
                            try {
                                return json_decode($responseBody, $associative);
                            } catch (Exception) {
                                throw new Exception('Failed to json decode api response: '.$result);
                            }
                        default:
                            return $responseBody;
                    }
                }
                return $responseBody;
            default:
                throw new Exception("CURL failed: \nurl: ".$url." \nmethod: '.$method.' \ncurl error number: ".$errorNumber." \n http code: ".$httpCode);
        }
    }
}