<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Helper\BodyTransform;
use Bespin\DocumentClient\Helper\Rest;
use Exception;

class Azure
{
    private string $key;
    private string $endpoint;

    public function __construct(string $azureEndpoint, string $azureKey)
    {
        $this->endpoint = $azureEndpoint;
        $this->key      = $azureKey;
    }

    /** @throws Exception */
    public function getInvoiceContent(string $fileName): object|null
    {
        $operationLocation = $this->sendFile($fileName);
        if ($operationLocation === null) {
            return null;
        }
        return $this->getParserResult($operationLocation);
    }

    /** @throws Exception */
    private function getParserResult(string $url): ?object
    {
        $count = 0;
        while (true) {
            $response = Rest::read(
                endpoint: $url,
                headers : $this->getRequestHeaders());
            if (is_object($response) && property_exists($response, 'status') && $response->status !== 'running') {
                break;
            }
            $count++;
            if ($count > 30) {
                break;
            }
            sleep(1);
        }
        if (is_object($response)) {
            return $response;
        }
        return null;
    }

    /**
     * @return array<string>
     */
    private function getRequestHeaders(): array
    {
        return [
            'Content-Type: application/json',
            'Ocp-Apim-Subscription-Key: '.$this->key
        ];
    }

    /** @throws Exception */
    private function sendFile(string $fileName): ?string
    {
        $fileContents = file_get_contents($fileName);
        if ($fileContents === false) {
            return null;
        }
        $base64PdfContent = base64_encode($fileContents);
        $restResult       = Rest::curl(
            endpoint     : $this->endpoint,
            location     : '/formrecognizer/documentModels/prebuilt-invoice:analyze?api-version=2023-07-31',
            method       : 'POST',
            parameters   : ['base64Source' => $base64PdfContent],
            headers      : $this->getRequestHeaders(),
            transform    : BodyTransform::JSON_ENCODE,
            returnHeaders: true
        );
        if (!is_array($restResult)) {
            return null;
        }
        [$headers] = $restResult;
        if (array_key_exists('operation-location', $headers)) {
            return $headers['operation-location'];
        }
        return null;
    }
}
