<?php

namespace Bespin\DocumentClient\Helper;

use DateTime;

class PdfMetadata
{
    public ?string   $title            = null;
    public ?string   $author           = null;
    public ?string   $subject          = null;
    public ?string   $language         = null;
    public ?string   $creator          = null;
    public ?string   $producer         = null;
    /** @var array<string>  */
    public array     $keywords         = [];
    public ?DateTime $creationDate     = null;
    public ?DateTime $modificationDate = null;

    public function getMetadata(): ?string
    {
        $result = [];
        foreach (['Title', 'Author', 'Subject', 'Producer', 'Language', 'Creator'] as $property) {
            if ($this->{strtolower($property)} !== null) {
                $result['/'.$property] = $this->{strtolower($property)};
            }
        }
        if (!empty($this->keywords)) {
            $result['/Keywords'] = implode(' ', $this->keywords);
        }
        if ($this->creationDate !== null) {
            $result['/CreationDate'] = self::getDateFormat($this->creationDate);
        }
        if ($this->modificationDate !== null) {
            $result['/ModDate'] = self::getDateFormat($this->modificationDate);
        }
        if (!empty($result)) {
            $json = json_encode($result);
            if ($json !== false) {
                return $json;
            }
        }
        return null;
    }

    public static function create(string $json): ?self
    {
        $decoded = json_decode($json, true);
        $result  = new self();
        $atLeastOnePropertyFound = false;
        if (is_iterable($decoded)) {
            foreach ($decoded as $property => $value) {
                if (is_string($property)) {
                    $convertedProperty = strtolower(ltrim($property, '/'));
                    switch ($convertedProperty) {
                        case 'moddate':
                            $result->modificationDate = self::getDate($value);
                            $atLeastOnePropertyFound = true;
                            break;
                        case 'creationdate':
                            $result->creationDate = self::getDate($value);
                            $atLeastOnePropertyFound = true;
                            break;
                        case 'keywords':
                            $result->keywords = explode(' ', $value);
                            $atLeastOnePropertyFound = true;
                            break;
                        default:
                            if (property_exists($result, $convertedProperty)) {
                                $result->{$convertedProperty} = $value;
                                $atLeastOnePropertyFound = true;
                            } else {
                                // TODO: log property name
                            }
                            break;
                    }
                }
            }
        }
        if ($atLeastOnePropertyFound === true) {
            return $result;
        }
        return null;
    }

    private static function getDateFormat(DateTime $date): string
    {
        return
            'D:'.
            $date->format('YmdHis').
            str_replace(':', "'", $date->format('P')).
            "'";
    }

    private static function getDate(string $dateString): ?DateTime
    {
        // Remove the leading 'D:' and the colons in the timezone offset
        $dateString = substr(str_replace("'", "", $dateString), 2);
        $dateTime = DateTime::createFromFormat('YmdHisO', $dateString);
        if ($dateTime === false) {
            return null;
        }
        return $dateTime;
    }
}