<?php

namespace Bespin\DocumentClient\Helper;

use Exception;

class PdfUtil
{
    /**
     * @return string
     * @throws Exception
     */
    private static function pdfUtil(): string
    {
        global $_composer_bin_dir;
        $binDir = $_composer_bin_dir;
        if ($binDir === null) {
            if (str_contains(__DIR__, 'vendor')) {
                $binDir = realpath(__DIR__.'/../../../../bin');
            } else {
                $binDir = realpath(__DIR__.'/../../bin');
            }
        }
        //$binDir = $_composer_bin_dir ?? realpath(__DIR__.'/../../../../');
        $binary = $binDir.'/pdfutil';
        if (file_exists($binary) && is_executable($binary)) {
            return $binary;
        }
        throw new Exception('PDF util not found');
    }

    /**
     * @param string $inputFileName
     * @param string $outputDirectory
     * @param array<string> $documentPages each array element creates a new document. the value can be comma separated pages and/or dashed ranges like this: 1-5,7,9,11-15
     * @param string $outputPrefix
     * @return array<string> returns an array of the files which were created
     * @throws Exception
     */
    public static function splitPdf(string $inputFileName, string $outputDirectory, array $documentPages, string $outputPrefix = 'document_'): array
    {
        // remove any whitespace in the page ranges as our pdf utility doesn't support this
        $documentPages = array_map(function ($value) {
            return preg_replace('/\s+/', '', $value);
        }, $documentPages);
        $arguments     = [
            self::pdfUtil(),
            'split',
            '--in '.$inputFileName,
            '--out '.$outputDirectory,
            '--pages '.implode(';', $documentPages),
            '--prefix '.$outputPrefix,
            '--format json'
        ];
        $result        = Shell::exec(
            $arguments
        );
        if ($result->getExitCode() === 0) {
            $json = json_decode($result->getStdOut());
            if (is_array($json)) {
                return $json;
            }
        }
        return [];
    }

    /**
     * @throws Exception
     */
    public static function mergePdfs(string $outputFilename, string ...$inputFilenames): bool
    {
        $arguments = [
            self::pdfUtil(),
            'merge',
            '--out '.$outputFilename,
            '--in '.implode(' ', $inputFilenames)
        ];
        $result    = Shell::exec(
            $arguments
        );
        return $result->getExitCode() === 0;
    }

    /**
     * @return array<string, string>
     * @throws Exception
     */
    public static function getMetadata(string $filename): array
    {
        $arguments = [
            self::pdfUtil(),
            'metadata',
            'get',
            '--in '.$filename,
            '--format json'
        ];
        $result    = Shell::exec(
            $arguments
        );
        if ($result->getExitCode() === 0) {
            $json = json_decode($result->getStdOut());
            if (is_array($json)) {
                return $json;
            }
        } else {
            throw new Exception($result->getStdErr(), $result->getExitCode());
        }
        return [];
    }

    /**
     * @param string $filename
     * @param array<string, string> $metadata
     * @param string|null $outputFilename
     * @return bool
     * @throws Exception
     */
    public static function setMetadata(string $filename, array $metadata, ?string $outputFilename): bool
    {
        $data = json_encode($metadata);
        if ($data !== false) {
            $arguments = [
                self::pdfUtil(),
                'metadata',
                'set',
                '--in '.$filename,
                '--data '.$data
            ];
            if ($outputFilename !== null) {
                $arguments[] = '--out '.$outputFilename;
            }
            $result = Shell::exec(
                $arguments
            );
            return $result->getExitCode() === 0;
        }
        return false;
    }
}