<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use Bespin\DocumentClient\Helper\Rest;
use Bespin\DocumentClient\DocumentType\Document as DocumentObject;
use CURLFile;
use Exception;

class Document
{
    public function __construct(private readonly string $url, private readonly string $bearer)
    {

    }

    /**
     * @param array<int> $groupIds
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function getDocument(int $documentId, int $userId, array $groupIds = [], bool $markAsRead = true): ?\Bespin\DocumentType\Document
    {
        $parameters = ['id' => $documentId, 'userId' => $userId];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        $parameters['markAsRead'] = $markAsRead;
        return \Bespin\DocumentType\Document::create(
            Rest::read(
                endpoint   : $this->url,
                location   : 'document',
                parameters : $parameters,
                headers    : ['AUTHORIZATION: '.$this->bearer],
                associative: true)
        );
    }

    /**
     * @param array<int> $groupIds
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function deleteDocument(int $documentId, int $userId, array $groupIds = []): void
    {
        $parameters = ['id' => $documentId, 'userId' => $userId];
        foreach ($groupIds as $groupId) {
            $parameters['groupIds'][] = $groupId;
        }
        Rest::delete(
            endpoint  : $this->url,
            location  : 'document',
            parameters: $parameters,
            headers   : ['AUTHORIZATION: '.$this->bearer]);
    }

    /**
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function createDocument(DocumentObject $document, bool $inspected = false): ?\Bespin\DocumentType\Document
    {
        if (!file_exists($document->getFile())) {
            throw new Exception('File '.$document->getFile().' does not exist');
        }
        $mimeType   = mime_content_type($document->getFile());
        $file       = new CURLFile($document->getFile(), $mimeType === false ? null : $mimeType);
        $parameters = [
            'file'         => $file,
            'owner'        => $document->getOwnerId(),
            'documentType' => $document->getType()->name()
        ];
        if ($document->getDocumentDate() !== null) {
            $parameters['documentDate'] = $document->getDocumentDate()->format('Y-m-d H:i:s');
        }
        if ($document->getSender() !== '') {
            $parameters['documentSender'] = $document->getSender();
        }
        if ($document->getReference() !== '') {
            $parameters['referenceNumber'] = $document->getReference();
        }
        if (!empty($document->getTags())) {
            $parameters['tags'] = json_encode($document->getTags());
        }
        if ($document->getAssigned() === false) {
            $parameters['assigned'] = 0;
        }
        $parameters['inspected'] = $inspected;
        //TODO: check if document has been uploaded successfully (md5 check), if yes, unlink local file, otherwise retry. Unlink file after x failed retries
        return \Bespin\DocumentType\Document::create(Rest::create(
            endpoint   : $this->url,
            location   : 'document',
            parameters : $parameters,
            headers    : ['AUTHORIZATION: '.$this->bearer],
            associative: true));
    }

    /**
     * @param array<int> $groupIds
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function updateDocument(
        \Bespin\DocumentType\Document $document,
        int                           $userId,
        array                         $groupIds = [],
        ?bool                         $assigned = null,
        ?bool                         $inspected = null
    ): ?\Bespin\DocumentType\Document
    {
        $parameters['documentType']    = $document->type?->value;
        $parameters['documentDate']    = $document->documentDate;
        $parameters['documentSender']  = $document->documentSender;
        $parameters['referenceNumber'] = $document->referenceNumber;
        if ($assigned !== null) {
            $parameters['assigned'] = $assigned ? 1 : 0;
        }
        $parameters['sourceId']      = $document->sourceId;
        $parameters['vendorEmailId'] = $document->vendorEmailId;
        $parameters['tags']          = $document->tags;
        $parameters['id']            = $document->documentId;
        $parameters['userId']        = $userId;
        $parameters['groupIds']      = $groupIds;
        if ($inspected !== null) {
            $parameters['inspected'] = $inspected;
        }

        return \Bespin\DocumentType\Document::create(Rest::update(
            endpoint  : $this->url,
            location  : 'document',
            parameters: $parameters,
            headers   : ['AUTHORIZATION: '.$this->bearer], associative: true));
    }

    /**
     * @param array<int> $groupIds
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public function setMetaData(int $documentId, string $metadata, int $userId, array $groupIds = []): void
    {
        $parameters = [
            'id'       => $documentId,
            'metadata' => $metadata,
            'userId'   => $userId,
            'groupIds' => $groupIds
        ];
        Rest::create(endpoint: $this->url, location: 'metadata', parameters: $parameters, headers: ['AUTHORIZATION: '.$this->bearer]);
    }
}
