<?php

namespace Bespin\DocumentClient\Helper;

use Bespin\DocumentClient\Exception\NotFoundException;
use Bespin\DocumentClient\Exception\PermissionDeniedException;
use CurlHandle;
use Exception;

class Rest
{
    public static bool $allowInsecure = false;

    /**
     * @param array<int|string, mixed> $parameters
     * @param array<string> $headers
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function create(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false, BodyTransform $transform = BodyTransform::NO): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            method     : 'POST',
            parameters : $parameters,
            headers    : $headers,
            associative: $associative,
            transform  : $transform
        );
    }

    /**
     * @param array<int|string, mixed> $parameters
     * @param array<string> $headers
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function read(string $endpoint, string $location = '', array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative
        );
    }

    /**
     * @param array<int|string, mixed> $parameters
     * @param array<string> $headers
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function update(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): mixed
    {
        return self::curl(
            endpoint   : $endpoint,
            location   : $location,
            method     : 'PUT',
            parameters : $parameters,
            headers    : $headers,
            associative: $associative,
            transform  : BodyTransform::URL_ENCODE
        );
    }

    /**
     * @param array<int|string, mixed> $parameters
     * @param array<string> $headers
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function delete(string $endpoint, string $location, array $parameters = [], array $headers = []): mixed
    {
        return self::curl(
            endpoint  : $endpoint,
            location  : $location,
            method    : 'DELETE',
            parameters: $parameters,
            headers   : $headers,
            transform : BodyTransform::URL_ENCODE);
    }

    /**
     * @param array<int|string, mixed> $parameters
     * @param array<string> $headers
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    public static function curl(
        string        $endpoint,
        string        $location = '',
        string        $method = 'GET',
        array         $parameters = [],
        array         $headers = [],
        bool          $associative = false,
        BodyTransform $transform = BodyTransform::NO,
        bool          $returnHeaders = false): mixed
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        $target = $endpoint.$location;
        if ($method === 'GET') {
            if (!empty($parameters)) {
                $target .= '?'.http_build_query($parameters);
            }
        } else {
            $parameters = match ($transform) {
                BodyTransform::JSON_ENCODE => json_encode($parameters),
                BodyTransform::URL_ENCODE  => http_build_query($parameters),
                default                    => $parameters,
            };
            curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
        }
        curl_setopt($ch, CURLOPT_URL, $target);

        if (self::$allowInsecure === true) {
            curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        }
        if (!empty($headers)) {
            if ($transform === BodyTransform::JSON_ENCODE) {
                $contentTypeFound = false;
                foreach ($headers as $header) {
                    if (stripos($header, 'Content-Type:') !== false) {
                        $contentTypeFound = true;
                        break;
                    }
                }
                if ($contentTypeFound === false) {
                    $headers[] = 'Content-Type: application/json';
                }
            }
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }

        $responseHeaders = [];
        curl_setopt($ch, CURLOPT_HEADERFUNCTION, function (CurlHandle $curl, string $header) use (&$responseHeaders) {
            $headers = explode(':', $header, 2);
            if (count($headers) === 2) {
                $responseHeaders[strtolower(trim($headers[0]))] = trim($headers[1]);
            }
            return strlen($header);
        });

        $responseBody = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        if (!is_string($responseBody)) {
            throw new Exception('Curl failed');
        }
        if ($returnHeaders === true) {
            return [$responseHeaders, self::parseResult($responseHeaders, $responseBody, $curlHttpCode, $errno, $endpoint.$location, $method, $associative)];
        }
        return self::parseResult($responseHeaders, $responseBody, $curlHttpCode, $errno, $endpoint.$location, $method, $associative);
    }

    /**
     * @param array<string, string> $responseHeaders
     * @throws NotFoundException
     * @throws PermissionDeniedException
     * @throws Exception
     */
    private static function parseResult(array $responseHeaders, string $responseBody, int $httpCode, int $errorNumber, string $url, string $method, bool $associative = false): mixed
    {
        $responseCategory = $httpCode - $httpCode % 100;
        switch ($responseCategory) {
            case 200:
                $header = new Header($responseHeaders);
                switch ($header->getContentType()) {
                    case ContentType::JSON:
                        if (empty($responseBody)) {
                            return '';
                        }
                        $json = json_decode($responseBody, $associative);
                        if ($json === null) {
                            throw new Exception(
                                self::getMessage(
                                    $url,
                                    $method, [
                                    'message'  => '',
                                    'response' => $responseBody,
                                    'headers'  => $responseHeaders
                                ]));
                        }
                        return $json;
                    default:
                        return $responseBody;
                }
            case 400:
                throw match ($httpCode) {
                    403     => new PermissionDeniedException(self::getMessage($url, $method, ['error' => $errorNumber, 'httpCode' => $httpCode])),
                    404     => new NotFoundException(self::getMessage($url, $method, ['error' => $errorNumber, 'httpCode' => $httpCode])),
                    default => new Exception(self::getMessage($url, $method, ['error' => $errorNumber, 'httpCode' => $httpCode])),
                };
            default:
                throw new Exception(self::getMessage($url, $method, ['error' => $errorNumber, 'httpCode' => $httpCode]));
        }
    }

    /**
     * @param string $url
     * @param string $method
     * @param array<string, mixed> $additionalParameters
     * @return string
     */
    private static function getMessage(string $url, string $method, array $additionalParameters = []): string
    {
        $parameters = [
            'url'    => $url,
            'method' => $method,
        ];
        $message    = json_encode($parameters + $additionalParameters);
        if ($message !== false) {
            return $message;
        }
        return 'Could not create rich error message for url: '.$url.'/'.$method.'. Json encode failed.';
    }
}