<?php

namespace Bespin\DocumentClient\Helper;

use Exception;

class Rest
{
    /** @throws Exception */
    public static function create(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): object|array
    {
        return self::curlWithPostFields(
            method        : 'POST',
            url           : $endpoint,
            location      : $location,
            parameters    : $parameters,
            headers       : $headers,
            httpBuildQuery: false,
            associative   : $associative
        );
    }

    /** @throws Exception */
    public static function read(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false): object|array
    {
        return self::get(
            url        : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative);
    }

    /** @throws Exception */
    public static function update(string $endpoint, string $location, array $parameters = [], array $headers = [], bool $associative = false)
    {
        return self::curlWithPostFields(
            method     : 'PUT',
            url        : $endpoint,
            location   : $location,
            parameters : $parameters,
            headers    : $headers,
            associative: $associative);
    }

    /** @throws Exception */
    public static function delete(string $endpoint, string $location, array $parameters = [], array $headers = [])
    {
        return self::curlWithPostFields(
            method        : 'DELETE',
            url           : $endpoint,
            location      : $location,
            parameters    : $parameters,
            headers       : $headers,
            httpBuildQuery: true);
    }

    /**
     * @throws Exception
     */
    private static function curlWithPostFields(string $method, string $url, string $location, array $parameters = [], array $headers = [], bool $httpBuildQuery = true, bool $associative = false)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url.$location);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $httpBuildQuery === true ? http_build_query($parameters) : $parameters);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        $result       = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        switch (($curlHttpCode - $curlHttpCode % 100) / 100) {
            case 2:
                try {
                    $json = json_decode($result, $associative);
                } catch (Exception) {
                    throw new Exception('Failed to json decode document service response: '.$result);
                }
                if (is_array($json) && array_key_exists('error', $json)) {
                    throw new Exception('Error in document service response: '.$json['error']);
                } elseif (is_object($json) && property_exists($json, 'error')) {
                    throw new Exception('Error in document service response: '.$json->error);
                }
                return $json;
            default:
                throw new Exception("CURL failed: \nurl: ".$url.$location." \nmethod: ".$method." \ncurl error number: ".$errno." \n http code: ".$curlHttpCode);
        }
    }

    /** @throws Exception */
    private static function get(string $url, string $location, array $parameters = [], array $headers = [], bool $associative = false): object|array
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url.$location.(!empty($parameters) ? '?'.http_build_query($parameters) : ''));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        if (!empty($headers)) {
            curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        }
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        $result       = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return match (($curlHttpCode - $curlHttpCode % 100) / 100) {
            2       => json_decode($result, $associative),
            default => throw new Exception("CURL failed: \nurl: ".$url.$location." \nmethod: GET \ncurl error number: ".$errno." \n http code: ".$curlHttpCode),
        };
    }
}