<?php

namespace Bespin\DocumentClient;

use Bespin\DocumentClient\Helper\StringFile;
use Exception;
use DateTime;
use DateTimeInterface;
use CURLFile;

class Document
{
    private string $url;
    private string $bearer;

    public function __construct(string $url, string $bearer)
    {
        $this->url    = $url;
        $this->bearer = $bearer;
    }

    /** @throws Exception */
    public function updateDocument(int $documentId, ?bool $assigned = null, ?DateTime $documentDate = null, ?string $referenceNumber = '')
    {

        if ($assigned !== null) {
            $params['assigned'] = $assigned ? 1 : 0;
        }
        if ($documentDate !== null) {
            $params['date'] = $documentDate->format('Y-m-d H:i:s');
        }
        if ($referenceNumber !== null) {
            $params['referenceNumber'] = $referenceNumber;
        }
        if (empty($params)) {
            return [];
        }
        $params['id'] = $documentId;
        return $this->put('document', $params);
    }

    /** @throws Exception */
    public function getUnassignedDocuments(): array
    {
        $documents = $this->get('unassigned');
        if (property_exists($documents, 'files') && is_array($documents->files)) {
            return $documents->files;
        }
        throw new Exception('Error retrieving undefined files from document service');
    }

    /** @throws Exception */
    public function getUnassignedDocumentsByType(string ...$types): array
    {
        $tags = $this->get('unassigned', ['types' => $types]);
        if (property_exists($tags, 'files') && is_array($tags->files)) {
            return $tags->files;
        }
        throw new Exception('Error retrieving undefined files from document service');
    }

    /** @throws Exception */
    public function getDocuments(?DateTimeInterface $from = null, ?DateTimeInterface $to = null, string ...$types): array
    {
        $parameters['types'] = $types;
        if ($from !== null) {
            $parameters['from'] = $from->format('Y-m-d');
        }
        if ($to !== null) {
            $parameters['to'] = $to->format('Y-m-d');
        }
        $tags = $this->get('documents', $parameters);
        if (property_exists($tags, 'files') && is_array($tags->files)) {
            return $tags->files;
        }
        return [];
    }

    /** @throws Exception */
    public function getDocument(int $documentId): object
    {
        return $this->get('document', ['id' => $documentId]);
    }

    /** @throws Exception */
    public function getDocumentMetaData(int $documentId): object
    {
        return $this->get('metadata', ['id' => $documentId]);
    }

    /** @throws Exception */
    public function getTags(): array
    {
        $tags = $this->get('tag');
        if (property_exists($tags, 'tags') && is_array($tags->tags)) {
            return $tags->tags;
        }
        throw new Exception('Error retrieving tags from document service');
    }

    /** @throws Exception */
    public function getDocumentContent(int $documentId): string
    {
        $document = $this->getDocument($documentId);
        return file_get_contents($document->url);
    }

    /** @throws Exception */
    public function postDocumentObject(DocumentType\Document $document): ?int
    {
        if (!file_exists($document->getFile())) {
            throw new Exception('File '.$document->getFile().' does not exist');
        }
        $file   = new CURLFile($document->getFile(), mime_content_type($document->getFile()));
        $result = $this->postDocument($file, $document->getType()->name(), $document->getSender(), $document->getReference(), $document->getDocumentDate(), $document->getTags(), $document->getAssigned());
        return $result->id ?? null;
    }

    /** @throws Exception */
    public function uploadDocumentContent(string $content, string $type, string $sender = '', string $reference = '', ?DateTime $date = null): object
    {
        $file = new StringFile($content);
        return $this->postDocument($file->getCurlFile(), $type, $sender, $reference, $date);
    }

    /** @throws Exception */
    private function postDocument(CURLFile $file, string $type, string $sender = '', string $reference = '', ?DateTime $date = null, array $tags = [], bool $assigned = true): object
    {
        $params = [
            'file'         => $file,
            'documentType' => $type
        ];
        if (!is_writable('/var/www/html/upload')) {
            throw new Exception('Path /var/www/html/upload is not writable');
        }
        if ($date !== null) {
            $params['date'] = $date->format('Y-m-d H:i:s');
        }
        if ($sender !== '') {
            $params['sender'] = $sender;
        }
        if ($reference !== '') {
            $params['referenceNumber'] = $reference;
        }
        if (!empty($tags)) {
            $params['tags'] = json_encode($tags);
        }
        if ($assigned === false) {
            $params['assigned'] = 0;
        }
        //TODO: check if document has been uploaded successfully (md5 check), if yes, unlink local file, otherwise retry. Unlink file after x failed retries
        $result = $this->post('document', $params, false);
        if (!is_object($result)) {
            file_put_contents('/var/www/html/log/bar.txt', $result);
        }
        return $result;
    }

    /** @throws Exception */
    private function curlWithPostFields(string $method, string $location, array $parameters = [], bool $httpBuildQuery = true)
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->url.$location);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method);
        if ($httpBuildQuery) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, http_build_query($parameters));
        } else {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $parameters);
        }
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [$this->bearer]);
        $result       = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        switch (($curlHttpCode - $curlHttpCode % 100) / 100) {
            case 2:
                try {
                    $json = json_decode($result);
                } catch (Exception) {
                    throw new Exception('Failed to json decode document service response: '.$result);
                }
                if (is_array($json) && array_key_exists('error', $json)) {
                    throw new Exception('Error in document service response: '.$json['error']);
                } elseif (is_object($json) && property_exists($json, 'error')) {
                    throw new Exception('Error in document service response: '.$json->error);
                }
                return $json;
            default:
                throw new Exception("CURL failed: \nurl: ".$this->url.$location." \nmethod: ".$method." \ncurl error number: ".$errno." \n http code: ".$curlHttpCode);
        }
    }

    /** @throws Exception */
    private function post(string $location, array $parameters = [], bool $httpBuildQuery = true)
    {
        return $this->curlWithPostFields('POST', $location, $parameters, $httpBuildQuery);
    }

    /** @throws Exception */
    private function put(string $location, array $parameters = [])
    {
        return $this->curlWithPostFields('PUT', $location, $parameters);
    }

    /** @throws Exception */
    private function get(string $method, array $parameters = []): object
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $this->url.$method.(!empty($parameters) ? '?'.http_build_query($parameters) : ''));
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [$this->bearer]);

        $result       = curl_exec($ch);
        $errno        = curl_errno($ch);
        $curlHttpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        return match (($curlHttpCode - $curlHttpCode % 100) / 100) {
            2       => json_decode($result),
            default => throw new Exception("CURL failed: \nurl: ".$this->url." \nmethod: ".$method." \ncurl error number: ".$errno." \n http code: ".$curlHttpCode),
        };
    }
}
