<?php

namespace Bespin\ConfigurationProfile;

use Bespin\ConfigurationProfile\Helper\Data;
use Bespin\ConfigurationProfile\Helper\UUID;
use Bespin\ConfigurationProfile\Payload\PayloadInterface;
use Bespin\ConfigurationProfile\Payload\Security\PKCS12;

class ConfigurationProfile
{
    private readonly string $uuid;
    /** @var PayloadInterface[] */
    private array $payloads = [];

    private const indent = 2;

    /**
     * @param string $id A reverse-DNS style identifier (com.example.myprofile, for example) that identifies the profile. This string is used to determine whether a new profile should replace an existing one or should be added.
     * @param string|null $displayName Optional. A human-readable name for the profile. This value is displayed on the Detail screen. It does not have to be unique.
     * @param string|null $description Optional. A description of the profile, shown on the Detail screen for the profile. This should be descriptive enough to help the user decide whether to install the profile.
     * @param string|null $organization Optional. A human-readable string containing the name of the organization that provided the profile.
     * @param ?bool $removalDisallowed Optional. Supervised only. If present and set to true, the user cannot delete the profile (unless the profile has a removal password and the user provides it)
     */
    public function __construct(
        private readonly string  $id,
        private readonly ?string $displayName,
        private readonly ?string $description,
        private readonly ?string $organization,
        private readonly ?bool   $removalDisallowed = null
    )
    {
        $this->uuid = UUID::get();
    }

    public function addPayload(PayloadInterface $payload): void
    {
        $this->payloads[] = $payload;
    }

    public function get(): string
    {
        $head = '<?xml version="1.0" encoding="UTF-8"?>
<!DOCTYPE plist PUBLIC "-//Apple//DTD PLIST 1.0//EN" "http://www.apple.com/DTDs/PropertyList-1.0.dtd">
<plist version="1.0">'.PHP_EOL;
        $foot = '</plist>';
        return $head.$this->generate($this->getProfile()).$foot;
    }

    private function getProfile(): object
    {
        $payloadContents = [];
        foreach ($this->payloads as $payload) {
            $payload->setConfigurationProfile($this);
            $payloadContents[] = $payload->getPayload();
        }

        $profile = [
            'PayloadContent'      => $payloadContents,
            'PayloadDisplayName'  => $this->displayName,
            'PayloadDescription'  => $this->description,
            'PayloadOrganization' => $this->organization,
            'PayloadIdentifier'   => $this->id,
            'PayloadType'         => 'Configuration',
            'PayloadUUID'         => $this->uuid,
            'PayloadVersion'      => 1,
        ];
        if (!empty($payloadContents)) {
            $profile['PayloadContent'] = $payloadContents;
        }
        if ($this->removalDisallowed !== null) {
            $profile['PayloadRemovalDisallowed'] = $this->removalDisallowed;
        }
        ksort($profile);
        return (object)$profile;
    }

    public function getOrganization(): string
    {
        return $this->organization;
    }

    public function getCertificateUUID(): string
    {
        foreach ($this->payloads as $payload) {
            if ($payload instanceof PKCS12) {
                return $payload->getUUID();
            }
        }
        return '';
    }

    private function generate(mixed $payload, $level = 0): string
    {
        $result = '';
        switch (true) {
            case ($payload instanceof Data):
                $result .= str_repeat(' ', $level * self::indent).'<data>'.$payload->get().'</data>'.PHP_EOL;
                break;
            case is_object($payload):
                $result .= str_repeat(' ', $level * self::indent).'<dict>'.PHP_EOL;
                $level++;
                foreach ($payload as $key => $payloadValue) {
                    $result .= str_repeat(' ', $level * self::indent).'<key>'.$key.'</key>'.PHP_EOL;
                    $result .= $this->generate($payloadValue, $level);
                }
                $level--;
                $result .= str_repeat(' ', $level * self::indent).'</dict>'.PHP_EOL;
                break;
            case is_array($payload):
                $result .= str_repeat(' ', $level * self::indent).'<array>'.PHP_EOL;
                $level++;
                foreach ($payload as $payloadValue) {
                    $result .= $this->generate($payloadValue, $level);
                }
                $level--;
                $result .= str_repeat(' ', $level * self::indent).'</array>'.PHP_EOL;
                break;
            case is_int($payload):
                $result .= str_repeat(' ', $level * self::indent).'<integer>'.$payload.'</integer>'.PHP_EOL;
                break;
            case is_bool($payload):
                $result .= str_repeat(' ', $level * self::indent).($payload === true ? '<true/> ' : '<false/>').PHP_EOL;
                break;
            case is_string($payload):
                $result .= str_repeat(' ', $level * self::indent).'<string>'.$payload.'</string>'.PHP_EOL;
                break;
            case is_float($payload):
                $result .= str_repeat(' ', $level * self::indent).'<real>'.$payload.'</real>'.PHP_EOL;
                break;
        }
        return $result;
    }
}
