<?php

namespace Bespin\Auth;

use Bespin\Auth\Authentication\Cognito;
use Bespin\Auth\Authentication\CognitoUser;
use byteShard\Authentication\IdentityProviderInterface;
use byteShard\Authentication\UserDataInterface;
use byteShard\Database;
use byteShard\Debug;
use byteShard\Environment;
use byteShard\Exception;
use Firebase\JWT\JWK;
use Firebase\JWT\JWT;
use JetBrains\PhpStorm\NoReturn;

class CognitoAuthentication implements IdentityProviderInterface
{

    private array $publicKeys;

    public function __construct(
        public readonly Environment $environment,
    ) {
    }

    /**
     * @return bool
     * @throws Exception
     */
    public function authenticate(): bool
    {
        $idToken = Cognito::getIdToken();
        if ($idToken !== false) {
            $tokenParts = explode('.', $idToken);
            $publicKeys = $this->getPublicKeys($tokenParts[1]);
            $userData   = $this->getJWTData($idToken, $publicKeys);
            return $userData !== null;
        }
        return false;
    }

    /**
     * @throws Exception
     */
    private function getISS(string $claims): ?string
    {
        $decodedClaims = JWT::jsonDecode(JWT::urlsafeB64Decode($claims));
        if (isset($decodedClaims->iss)) {
            $iss = $decodedClaims->iss;
            if ($this->validateISS($iss)) {
                return $iss;
            } else {
                throw new Exception('Invalid tenant');
            }
        }
        return null;
    }

    private function validateISS(string $iss): bool
    {
        $parsedUrl    = parse_url($iss);
        $path         = $parsedUrl['path'];
        $pathSegments = explode('/', $path);
        $userPoolId   = end($pathSegments);
        return Database::getSingle('SELECT tenant FROM Tenants WHERE userPool=:userPool', ['userPool' => $userPoolId]) !== null;
    }

    private function getJWTData(string $token, array $publicKey): ?object
    {
        return JWT::decode($token, $publicKey);
    }

    #[NoReturn]
    public function logout()
    {
        $this->environment->printLoginCallback('login');
        if (session_status() === PHP_SESSION_ACTIVE) {
            session_destroy();
            session_unset();
        }
        header("Refresh:0");
        exit;
    }

    /**
     * @throws Exception
     */
    public function getUserData(): UserDataInterface
    {
        $idToken = Cognito::getIdToken();
        if ($idToken !== false) {
            $idTokenDecoded = Cognito::getDecodedIdToken($idToken);
            if ($idTokenDecoded !== null) {
                Debug::debug(var_export($idTokenDecoded, true));
                $user = CognitoUser::UserFromAttributes($idTokenDecoded);
                Debug::debug(var_export($user, true));
                return $user;
            }
        }
        throw new Exception('Could not retrieve userdata');
    }

    /**
     * @param string $claims
     * @return array|null
     * @throws Exception
     */
    protected function getPublicKeys(string $claims): ?array
    {
        if (empty($this->publicKeys)) {
            $iss = $this->getISS($claims);
            if ($iss !== null) {
                $url              = $iss.'/.well-known/jwks.json';
                $publicKeys       = $this->getPublicKey($url);
                $this->publicKeys = JWK::parseKeySet($publicKeys);
            } else {
                return null;
            }
        }
        return $this->publicKeys;
    }

    private function getPublicKey(string $url): ?array
    {
        $ch = curl_init();
        curl_setopt($ch, CURLOPT_URL, $url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        $result = curl_exec($ch);
        curl_close($ch);
        return json_decode($result, true);
    }
}
