<?php

namespace Bespin\Auth\Authentication;

use Bespin\Auth\Authentication\Model\Helper;
use byteShard\Database;
use byteShard\Debug;
use Exception;

class CognitoUser implements CognitoUserInterface
{

    private ?int   $userId;
    private string $userName;
    private bool   $serviceAccount;
    private array  $groups      = [];
    private array  $attributes  = [];
    private string $email       = '';
    private string $fullName    = '';
    private string $phoneNumber = '';
    private string $apiKey      = '';

    public function __construct(?int $userId = null, string $userName = '', bool $serviceAccount = false)
    {
        $this->userId         = $userId;
        $this->userName       = $userName;
        $this->serviceAccount = $serviceAccount;
    }


    /**
     * @throws \byteShard\Exception
     * @throws Exception
     */
    public static function UserFromAttributes(object $attributes): self
    {
        Debug::debug(var_export($attributes, true));

        $user   = new User($attributes);
        $userId = self::updateOrAddUserWhenNeeded($user);
        if ($userId === null) {
            throw new Exception('Could not get/create user');
        }
        $userObj = new self(
            $userId,
            $user->getAttribute(CognitoProperties::user)
        );
        $userObj->setEmail($user->getAttribute(CognitoProperties::email));
        if ($user->getAttribute(CognitoProperties::name) !== null) {
            $userObj->setFullName($user->getAttribute(CognitoProperties::name));
        }
        if ($user->getAttribute(CognitoProperties::groups) !== null) {
            $userObj->setGroups($user->getAttribute(CognitoProperties::groups));
        }
        if ($user->getAttribute(CognitoProperties::apikey) !== null) {
            $userObj->setApiKey($user->getAttribute(CognitoProperties::apikey));
        }
        return $userObj;
    }

    /**
     * @throws \byteShard\Exception
     */
    private static function updateOrAddUserWhenNeeded(User $user): ?int
    {
        $userMail = $user->getAttribute(CognitoProperties::email);
        $objectId = $user->getAttribute(CognitoProperties::user);
        if (empty($userMail)) {
            Debug::error('Error: Mail is empty for user, aborting: '.$objectId);
            return null;
        }
        $userFromDB = Database::getSingle('SELECT userId FROM Users WHERE user=:username', ['username' => $objectId]);
        if ($userFromDB === null) {
            $userFromDB = Database::getSingle('SELECT userId FROM Users WHERE mail=:mail', ['mail' => $userMail]);
            if ($userFromDB !== null) {
                self::setUserObjectId($userFromDB->userId, $objectId);
            }
        }
        return self::addOrUpdateUser($user);
    }

    /**
     * @param User $user
     * @return int|null
     * @throws \byteShard\Exception
     */
    private static function addOrUpdateUser(User $user): ?int
    {
        $userMail = $user->getAttribute(CognitoProperties::email);
        $objectId = $user->getAttribute(CognitoProperties::user);
        $users    = Database::getArray('SELECT userId FROM Users WHERE user=:objectID OR mail=:mail', ['objectID' => $objectId, 'mail' => $userMail]);
        if (count($users) > 1) {
            Debug::error('Multiple Users found, abort '.$objectId.' '.$userMail);
            return null;
        }
        $parameters['mail'] = $userMail;
        $parameters['user'] = $objectId;
        if (count($users) === 1) {
            Database::update('UPDATE Users SET '.Helper::set($parameters).' WHERE userId=:userId', Helper::push($parameters, ['userId' => $users[0]->userId]));
            return $users[0]->userId;
        } else {
            $result = Database::insert('INSERT INTO Users ('.Helper::insertColumns($parameters).') VALUES ('.Helper::insertValues($parameters).')', $parameters);
            if (is_int($result)) {
                return $result;
            }
        }
        return null;
    }

    /**
     * @throws \byteShard\Exception
     */
    public static function setUserObjectId(int $userId, $objectId): void
    {
        Database::update(
            'UPDATE Users SET user=:objectId WHERE userId=:userId',
            [
                'objectId' => $objectId,
                'userId'   => $userId
            ]
        );
    }

    public function setGroups(array $groups): void
    {
        $this->groups = $groups;
    }

    public function setAttributes(array $attributes): void
    {
        $this->attributes = $attributes;
    }

    public function setEmail(string $email): void
    {
        $this->email = $email;
    }

    public function setFullName(string $fullName): void
    {
        $this->fullName = $fullName;
    }

    public function setPhoneNumber(string $phoneNumber): void
    {
        $this->phoneNumber = $phoneNumber;
    }

    public function getEmail(): string
    {
        return $this->email;
    }

    public function getFullName(): string
    {
        return $this->fullName;
    }

    public function getPhoneNumber(): string
    {
        return $this->phoneNumber;
    }

    public function getGroups(): array
    {
        return $this->groups;
    }

    public function getAttributes(): array
    {
        return $this->attributes;
    }

    /**
     * the user id is the internal database id
     * @return ?int
     */
    public function getUserId(): ?int
    {
        return $this->userId;
    }

    /**
     * the username is the name the user uses to log in
     * @return string
     */
    public function getUsername(): string
    {
        return $this->userName;
    }

    public function isServiceAccount(): bool
    {
        return $this->serviceAccount;
    }

    public function getApiKey(): string
    {
        return $this->apiKey;
    }

    public function setApiKey(string $apiKey): void
    {
        $this->apiKey = $apiKey;
    }
}