<?php

namespace Bespin\Calendar;

use byteShard\Password;
use DateTime;
use Exception;

class WebDavCalendar extends Calendar
{
    protected string   $calendarUrl;
    protected string   $username;
    protected Password $password;

    public function __construct(string $calendarUrl, string $username, Password $password)
    {
        $this->calendarUrl = $calendarUrl;
        $this->username    = $username;
        $this->password    = $password;
    }

    /**
     * @throws Exception
     */
    public function addEntry(string $summary, DateTime $startDate, DateTime $endDate, string $email, string $name, string $description): string
    {
        $uid        = uniqid();
        $icsContent = new Event($uid, $summary, $startDate, $endDate, $email, $name, $description);
        $fileName   = $uid.'.ics';

        $this->webDavRequest(WebDavMethodEnum::PUT, $fileName, $icsContent->createICalObject());
        return $uid;
    }

    /**
     * @throws Exception
     */
    public function deleteEntry(string $entryId): bool
    {
        $fileName = $entryId.'.ics';
        $response = $this->webDavRequest(WebDavMethodEnum::DELETE, $fileName);
        return is_bool($response) ? $response : true;
    }

    /**
     * @throws Exception
     */
    private function webDavRequest(WebDavMethodEnum $method, string $path = '', string $body = ''): bool|string
    {
        $url     = $this->calendarUrl.$path;
        $headers = [
            'Content-Type: text/calendar; charset=utf-8',
            'Depth: 1',
        ];

        $ch = curl_init($url);
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, $method->name);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($ch, CURLOPT_USERPWD, $this->username.':'.$this->password);

        if ($body) {
            curl_setopt($ch, CURLOPT_POSTFIELDS, $body);
        }

        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);

        if ($httpCode >= 400) {
            throw new Exception('WebDAV request failed with HTTP code: '.$httpCode);
        }

        return $response;
    }
}
